/** Klasse Obstacle
 * 
 * 	Definiert Hindernisse in Form eines geraden Zylinders mit Mittelachse parallel zur z-Achse
 * 	
 * Attribute:
 * 
 * 	location {float[3]} Mittelpunkt der Grundflche des Zylinders
 * 	height {float} Hhe des Zylinders (bzw. Abstand der Mittelpunkte von Grund- und Deckflche)
 * 	radius {float} Radius des Zylinders
 * 	name {String} Bezeichnung des Hindernisses umd es von anderen unterscheidbar zu machen - Alternative zur ObjectID
 * 	gridCell {gridCell} Referenz auf die Zelle im Gitternetz in der sich location befindet
 *
 */




/**
 * Obstacle(location, height, radius, name) : Obstacle
 * 
 * 	Konstruktor und zugleich Definition der Klasse Obstacle
 * 
 * Parameter:
 * 
 * 	location {float[3]} siehe Klassenattribute
 * 	height {float} siehe Klassenattribute
 * 	radius {float} siehe Klassenattribute
 * 	name {String} siehe Klassenattribute
 * 
 * Rckgabewert:
 * 
 * {Obstacle} neues Obstacle Objekt
 * 
 */
function Obstacle(location, height, radius, name){
	this.location = location;
	this.height = height;
	this.radius = radius;
	this.name = name;
	this.gridCell; //wird erst bei Ausfhrung von "placeIntoGrid()" gesetzt
};

/**
 * Methode placeIntoGrid() : void
 * 
 * 	Markiert die Zellen im Gitternetz in denen das Hindernis sichtbar sein knnte,
 * 	genauer wird der Einfachheit halber eine Art quadrischer Hllkrper um den Zylinder gezogen
 * 
 * Klassenmethode zu Obstacle
 * 
 */
Obstacle.prototype.placeIntoGrid = function(){
	var start = [];
	var end = [];
	var offSet = theGrid.gridBorderFar;
	
	//Radius und Hhe in "Gridzellenbreite" und aufgerundet
	var radCeil = Math.ceil(this.radius / theGrid.gridCellWidth);
	var heightCeil = Math.ceil(this.height / theGrid.gridCellWidth);
	
	var gridPos = [];
	//"Fupunkt des Zylinders in Grid einsortieren
	gridPos[0] = Math.floor((this.location[0] - offSet[0])/theGrid.gridCellWidth);
	gridPos[1] = Math.floor((this.location[1] - offSet[1])/theGrid.gridCellWidth);
	gridPos[2] = Math.floor((this.location[2] - offSet[2])/theGrid.gridCellWidth);
	 
	theGrid.data[gridPos[0]][gridPos[1]][gridPos[2]].addObstacle(this);
	this.gridCell = theGrid.data[gridPos[0]][gridPos[1]][gridPos[2]];
	
	
	//Bereich um gridPos herum bestimmen, in dem das Hindernis eingetragen werden soll
	start[0] = gridPos[0] - radCeil;
	start[1] = gridPos[1] - radCeil;
	start[2] = 0; //0, da sich location am Boden des Zylinders befindet
	
	end[0] = gridPos[0] + radCeil;
	end[1] = gridPos[1] + radCeil;
	end[2] = heightCeil;
	
	//Abmessungen des Hllquaders auf gltigen Bereich eingrenzen
	//d.h. Verweise nur in Zellen eintragen, die auch wirklich existieren
	for(var i=0; i<3; i++){
		if(start[i]<0){
			start[i] = 0;
		}
		else if(start[i]>=theGrid.gridLength[i]){
			start[i] = theGrid.gridLength[i]-1;
		}
		
		if(end[i]<0){
			end[i] = 0;
		}
		else if(end[i]>=theGrid.gridLength[i]){
			end[i] = theGrid.gridLength[i]-1;
		}
	}
	
	//Verweis in Grid- Zellen eintragen
	for(var i=start[0]; i<=end[0]; i++){
		for(var j=start[1]; j<=end[1]; j++){
			for(var k=start[2]; k<=end[2]; k++){
				theGrid.data[i][j][k].addObstacle(this);			
			}
		}
	}
};


/**
 * Methode createVertexPositionBuffer() : ArrayBuffer
 * 
 * 	Aus den Abmessungen des Hindernisses werden die Eckpunkte der graphischen Reprsentation errechnet,
 * 	diese wird dann als Quader dargestellt.
 * 
 * Rckgabewert:
 * 
 * {ArrayBuffer} Zum Zeichnen bentigter ArrayBuffer 
 * 
 * Klassenmethode zu Obstacle
 * 
 */
Obstacle.prototype.createVertexPositionBuffer = function(){
	var vertexPositionBuffer;
	var loc = this.location;
	var height = this.height;
	var radius = this.radius;
	vertexPositionBuffer = gl.createBuffer();
    gl.bindBuffer(gl.ARRAY_BUFFER, vertexPositionBuffer);
    var vertices = [
                    loc[0] + radius, loc[1], loc[2], //Grundflche
                    loc[0], loc[1] + radius, loc[2],
                    loc[0] - radius, loc[1], loc[2],
                    loc[0], loc[1] - radius, loc[2],
                    
                    loc[0] + radius, loc[1], loc[2], //Seitenflche 1
                    loc[0], loc[1] + radius, loc[2],
                    loc[0], loc[1] + radius, loc[2] + height,
                    loc[0] + radius, loc[1], loc[2] + height,
                    
                    loc[0], loc[1] + radius, loc[2], //Seitenflche 2
                    loc[0] - radius, loc[1], loc[2],
                    loc[0] - radius, loc[1], loc[2] + height,
                    loc[0], loc[1] + radius, loc[2] + height,
                    
                    loc[0] - radius, loc[1], loc[2], //Seitenflche 3
                    loc[0], loc[1] - radius, loc[2],
                    loc[0], loc[1] - radius, loc[2] + height,
                    loc[0] - radius, loc[1], loc[2] + height,
                    
                    loc[0], loc[1] - radius, loc[2], //Seitenflche 4
                    loc[0] + radius, loc[1], loc[2],
                    loc[0] + radius, loc[1], loc[2] + height,
                    loc[0], loc[1] - radius, loc[2] + height,
                    
                    loc[0] + radius, loc[1], loc[2] + height, //Deckflche
                    loc[0], loc[1] + radius, loc[2] + height,
                    loc[0] - radius, loc[1], loc[2] + height,
                    loc[0], loc[1] - radius, loc[2] + height
                 ];
	gl.bufferData(gl.ARRAY_BUFFER, new Float32Array(vertices), gl.STATIC_DRAW);
	vertexPositionBuffer.itemSize = 3;
	vertexPositionBuffer.numItems = 24;
	
	return vertexPositionBuffer;
};
